<?php

namespace App\Http\Controllers;

use App\Models\WhatsAppSettings;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class WhatsAppSettingsController extends Controller
{
    /**
     * Display WhatsApp settings
     */
    public function index(): Response
    {
        $currentCompany = auth()->user()->getCurrentCompany();
        $settings = WhatsAppSettings::getCurrent();
        
        return Inertia::render('administration/WhatsAppSettings', [
            'settings' => $settings,
            'currentCompany' => $currentCompany,
        ]);
    }

    /**
     * Store or update WhatsApp settings
     */
    public function store(Request $request): RedirectResponse
    {
        $currentCompany = auth()->user()->getCurrentCompany();
        if (!$currentCompany) {
            return redirect()->back()->with('error', 'No company selected');
        }
        
        $isActive = $request->boolean('is_active');
        $provider = $request->input('provider', 'twilio');
        
        $validated = $request->validate([
            'provider' => ['required', 'string', 'in:twilio,meta'],
            'api_key' => [$isActive ? 'required' : 'nullable', 'string', 'max:1000'], // Meta tokens can be long
            'api_secret' => [$provider === 'twilio' && $isActive ? 'required' : 'nullable', 'string', 'max:1000'],
            'account_sid' => [$provider === 'twilio' && $isActive ? 'required' : 'nullable', 'string', 'max:255'],
            'from_number' => [$isActive ? 'required' : 'nullable', 'string', 'max:20'],
            'is_active' => ['boolean'],
        ]);

        // Get or create settings for current company
        $settings = WhatsAppSettings::getForCompany($currentCompany->id);
        
        if ($settings) {
            // Update existing settings
            $settings->update([
                'provider' => $validated['provider'],
                'api_key' => $validated['api_key'] ?? '',
                'api_secret' => $validated['api_secret'] ?? '',
                'account_sid' => $validated['account_sid'] ?? '',
                'from_number' => $validated['from_number'] ?? '',
                'is_active' => $isActive,
            ]);
        } else {
            // Create new settings (only if active or if we have credentials)
            if ($isActive || !empty($validated['api_key']) || !empty($validated['from_number'])) {
                WhatsAppSettings::create([
                    'company_id' => $currentCompany->id,
                    'provider' => $validated['provider'],
                    'api_key' => $validated['api_key'] ?? '',
                    'api_secret' => $validated['api_secret'] ?? '',
                    'account_sid' => $validated['account_sid'] ?? '',
                    'from_number' => $validated['from_number'] ?? '',
                    'is_active' => $isActive,
                ]);
            }
        }

        return redirect()->back()->with('success', 'WhatsApp settings saved successfully');
    }

    /**
     * Update existing WhatsApp settings
     */
    public function update(Request $request, WhatsAppSettings $whatsAppSettings): RedirectResponse
    {
        $currentCompany = auth()->user()->getCurrentCompany();
        if (!$currentCompany) {
            return redirect()->back()->with('error', 'No company selected');
        }
        
        // Ensure the settings belong to the current company
        if ($whatsAppSettings->company_id !== $currentCompany->id) {
            abort(403, 'Unauthorized access to WhatsApp settings.');
        }
        
        $isActive = $request->boolean('is_active');
        $provider = $request->input('provider', 'twilio');
        
        $validated = $request->validate([
            'provider' => ['required', 'string', 'in:twilio,meta'],
            'api_key' => [$isActive ? 'required' : 'nullable', 'string', 'max:1000'], // Meta tokens can be long
            'api_secret' => [$provider === 'twilio' && $isActive ? 'required' : 'nullable', 'string', 'max:1000'],
            'account_sid' => [$provider === 'twilio' && $isActive ? 'required' : 'nullable', 'string', 'max:255'],
            'from_number' => [$isActive ? 'required' : 'nullable', 'string', 'max:20'],
            'is_active' => ['boolean'],
        ]);

        // If disabling WhatsApp, clear the credentials
        if (!$isActive) {
            $validated['api_key'] = '';
            $validated['api_secret'] = '';
            $validated['account_sid'] = '';
            $validated['from_number'] = '';
        }

        $whatsAppSettings->update($validated);

        return redirect()->back()->with('success', 'WhatsApp settings updated successfully');
    }

    /**
     * Delete WhatsApp settings
     */
    public function destroy(WhatsAppSettings $whatsAppSettings): RedirectResponse
    {
        $currentCompany = auth()->user()->getCurrentCompany();
        if (!$currentCompany) {
            return redirect()->back()->with('error', 'No company selected');
        }
        
        // Ensure the settings belong to the current company
        if ($whatsAppSettings->company_id !== $currentCompany->id) {
            abort(403, 'Unauthorized access to WhatsApp settings.');
        }
        
        $whatsAppSettings->delete();

        return redirect()->back()->with('success', 'WhatsApp settings deleted successfully');
    }
}
